using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Text;
using System.Windows.Forms;


namespace Campo_de_Batalla___Calculos_Previos {

    public partial class Form1 : Form {
        /*
         * Clases internas
         */
        class PlayerData {
            public int x;                      // Posicion en X
            public int y;                      // Posicion en Y
            public int kills;                  // Cantidad de tanques que ha eliminado
            public int deads;                  // Cantidad de veces que lo han eliminado
            public int direction;              // Direccion en la que se encuentra
            public int fights;                 // Cuantos enfretamientos lleva
            public int num;                    // Numero de jugador (en batalla)
            public string name;                // Nombre del participante
            public string program;             // Programa que utiliza
        }

        /* 
         * Constantes
         */
        const int NUM_PLAYERS_SCREEN = 4;      // Jugadores en pantalla al mismo tiempo
        const int NUM_INSTRUCTIONS =   6;      // Cantidad de instrucciones distintas
        const int NUM_DIR =            4;      // Canridad de direcciones (N, S, E, O)
        const int SHOT_RANGE =         5;      // Cuadros de alcanze del disparo
        const int GRID_SIZE =          7;      // Tamao de la cuadricula (en cuadros)
        const int WALL =              -1;      // Pared (para detectar colisiones)
        const int EMPTY =              0;      // Vacio (para detectar colisiones)
        const int DANGEROUS =          9;      // Un nmero "grande" para saber que la casilla es peligros (al colocar de nuevo un tanque)
        const string PROGRAM_FOLDER = "programs/";            // Carpeta con los programas
        const string ERROR_FILE =     "error.txt";            // Texto con los errores de ejecucion
        const string BATTLE_FILE =    "programs/precal.dat";  // Donde se guardan las acciones de las batallas
        const string PLAYERS_FILE =   "programs/players.txt"; // Texto con los jugadores y sus programas
        const string DATA_FILE =      "programs/datos.txt";   // Donde leen los jugadores la situacion actual en la batalla
        const string ACTION_FILE =    "programs/salida.txt";  // Donde escriben los jugadores su movimiento
        readonly int[,] startCoord = {         // Coordenadas en la que inician los tanques
            {2, GRID_SIZE}, {GRID_SIZE, GRID_SIZE -1}, 
            {GRID_SIZE -1, 1}, {1, 2}
        };
        readonly int[,] movement = {           // Movimientos (Este, Sur, Oeste, Norte)
            {1, 0}, {0, -1}, {-1, 0}, {0, 1}
        };
        readonly char[] instruction = {        // Instrucciones
            'A', 'R', 'I', 'D', 'S', 'N'
        };
        readonly char[] direction = {          // Direccion (en sentido horario, empezando por el "Este")
            'E', 'S', 'O', 'N'
        };

        /*
         * Variables
         */
        int numErrors;                             // Cantidad de errores
        int[,] matches;                         // Orden de los encuentros
        PlayerData[] players;                   // Jugadores participando (el arreglo se crea cuando sepamos cuantos son)
        PlayerData[] currPlayers =              // Jugadores activos
            new PlayerData[NUM_PLAYERS_SCREEN];
        int[,] collision =                      // Para revisar colisiones
            new int[GRID_SIZE + 2, GRID_SIZE + 2];


        /*
         * Constructor de la clase
         */
        public Form1() {
            InitializeComponent();

            // Ponemos la lista de los programas que pueden quedar fijos
            for (int i = 0; i< NUM_PLAYERS_SCREEN; ++i) {
                presetProgListBox.Items.Add(i+1, false);
            }
        }

        /*
         * Permutamos un arreglo aleatoreamente
         */
        void randomize_array(int[] array) {
            Random random = new Random(DateTime.Now.Millisecond);
            for (int i = 0; i < array.Length; ++i) {
                int pos = random.Next(array.Length);
                int temp = array[i];
                array[i] = array[pos];
                array[pos] = temp;
            }
        }

        /*
         * Obtenemos cuales van a ser los encuentros de forma aleatoria
         */
        void randomize_matches(int totalMatches) {
            // Vemos cuantos y cuales jugadores quedan fijos
            int presetCount = 0;
            bool[] presetChecked = new bool[NUM_PLAYERS_SCREEN];
            for (int i = 0; i < presetChecked.Length; ++i) {
                presetChecked[i] = presetProgListBox.GetItemChecked(i);
                if (presetChecked[i]) {
                    presetCount++;
                }
            }

            // Repartimos a los jugadores en arreglos de acuerdo a si estan fijos o no
            int setPos = 0;
            int nonPos = 0;
            int[] preset = new int[presetCount*(players.Length/NUM_PLAYERS_SCREEN)];
            int[] nonPreset = new int[players.Length -preset.Length];
            for (int i = 0; i < players.Length; ++i) {
                if (presetChecked[i % NUM_PLAYERS_SCREEN]) {
                    preset[setPos++] = i;
                } else {
                    nonPreset[nonPos++] = i;
                }
            }
            randomize_array(nonPreset);

            // Asignamos los encuentros
            setPos = 0;
            nonPos = 0;
            matches = new int[totalMatches, NUM_PLAYERS_SCREEN];
            for (int i = 0; i < totalMatches; ++i) {
                for (int j = 0; j < NUM_PLAYERS_SCREEN; ++j) {
                    if (presetChecked[j]) {
                        // Ponemos de los fijos
                        matches[i, j] = preset[setPos++];
                        if (setPos >= preset.Length) {
                            setPos = 0;
                        }
                    } else {
                        // Ponemos de los aleatorios
                        matches[i, j] = nonPreset[nonPos++];
                        if (nonPos >= nonPreset.Length) {
                            nonPos = 0;
                            randomize_array(nonPreset);
                        }
                    }
                }
            }
        }

        /*
         * Inicializamos la pelea
         */
        void initialize_match(int match, StreamWriter text, StreamWriter errorFile) {
            // Escogemos a los jugadores
            for (int i = 0; i < NUM_PLAYERS_SCREEN; ++i) {
                currPlayers[i] = players[matches[match, i]];
                currPlayers[i].num = i + 1;
                currPlayers[i].fights++;
            }

            // Ponemos los jugadores en el archivo de texto
            errorFile.WriteLine();
            for (int i = 0; i < NUM_PLAYERS_SCREEN - 1; ++i) {
                text.Write(matches[match, i] + 1 + " ");
                errorFile.Write(currPlayers[i].name + " vs. ");
            }
            text.WriteLine(matches[match, NUM_PLAYERS_SCREEN - 1] + 1);
            errorFile.WriteLine(currPlayers[NUM_PLAYERS_SCREEN - 1].name);

            // Valores iniciales de las colisiones
            Array.Clear(collision, 0, collision.Length);
            for (int i = 0; i <= GRID_SIZE + 1; ++i) {
                collision[0, i] = WALL;
                collision[GRID_SIZE + 1, i] = WALL;
                collision[i, 0] = WALL;
                collision[i, GRID_SIZE + 1] = WALL;
            }

            // Valores iniciales de los jugadores
            for (int i = 0; i < NUM_PLAYERS_SCREEN; ++i) {
                currPlayers[i].kills = 0;
                currPlayers[i].deads = 0;
                currPlayers[i].x = startCoord[i, 0];
                currPlayers[i].y = startCoord[i, 1];
                currPlayers[i].direction = i;
                collision[currPlayers[i].x, currPlayers[i].y] = i + 1;
            }
        }

        /*
         * Ponemos al tanque en una posicion segura (lo acaban de matar)
         */
        void respawn_tank(PlayerData player) {
            // Obtenemos una matriz con la distancia "manhattan" a los otros jugadores
            int[,] distance = new int[GRID_SIZE, GRID_SIZE];
            for (int i = 0; i < GRID_SIZE; ++i) {
                for (int j = 0; j < GRID_SIZE; ++j) {
                    distance[i, j] = 0;
                    for (int k = 0; k < NUM_PLAYERS_SCREEN; ++k) {
                        if (k != player.num -1) {
                            distance[i, j] += Math.Abs(currPlayers[k].x-1 - i) + Math.Abs(currPlayers[k].y-1 - j);
                        }
                    }
                }
            }

            // Obtenemos una matriz con la "peligrosidad" de cada cuadro de acuerdo a la posicion de los otros jugadores
            int[,] danger = new int[GRID_SIZE, GRID_SIZE];
            Array.Clear(danger, 0, danger.Length);
            // Hacemos que las casillas que estan en la misma fila o columna de los otros jugadores sean las mas peligrosas
            for (int k = 0; k < NUM_PLAYERS_SCREEN; ++k) {
                if (k != player.num -1) {
                    for (int i = 0; i < GRID_SIZE; ++i) {
                        danger[currPlayers[k].x -1, i] = DANGEROUS;
                        danger[i, currPlayers[k].y -1] = DANGEROUS;
                    }
                }
            }
            // Revisando a la derecha
            for (int i = GRID_SIZE - 2; i >= 0; --i) {
                for (int j = 0; j < GRID_SIZE; ++j) {
                    danger[i, j] = Math.Max(danger[i, j], danger[i + 1, j] - 1);
                }
            }
            // Revisando a la izquierda
            for (int i = 1; i < GRID_SIZE; ++i) {
                for (int j = 0; j < GRID_SIZE; ++j) {
                    danger[i, j] = Math.Max(danger[i, j], danger[i - 1, j] - 1);
                }
            }
            // Revisando abajo
            for (int i = 0; i < GRID_SIZE; ++i) {
                for (int j = GRID_SIZE - 2; j >= 0; --j) {
                    danger[i, j] = Math.Max(danger[i, j], danger[i, j + 1] - 1);
                }
            }
            // Revisando arriba
            for (int i = 0; i < GRID_SIZE; ++i) {
                for (int j = 1; j < GRID_SIZE; ++j) {
                    danger[i, j] = Math.Max(danger[i, j], danger[i, j -1] - 1);
                }
            }

            // Encontramos la nueva posicion del tanque
            int newX = 0;
            int newY = 0;
            int minDanger = DANGEROUS;
            int maxDistance = 0;
            for (int i = 0; i < GRID_SIZE; ++i) {
                for (int j = 0; j < GRID_SIZE; ++j) {
                    if ((minDanger > danger[i, j]) ||
                        ((minDanger == danger[i, j]) && (maxDistance < distance[i, j]))) {
                        newX = i;
                        newY = j;
                        minDanger = danger[i, j];
                        maxDistance = distance[i, j];
                    }
                }
            }
            collision[player.x, player.y] = EMPTY;
            player.x = newX + 1;
            player.y = newY + 1;
            collision[player.x, player.y] = player.num;
        }

        /*
         * Actualiza al tanque de acuerdo a la opcion que eligi
         */
        void update_actions(PlayerData player, char action, StreamWriter text) {
            int x = player.x;
            int y = player.y;
            int dX = movement[player.direction, 0];
            int dY = movement[player.direction, 1];
            PlayerData playerKilled;

            // Revisamos que accion se efectuo
            switch (action) {
                case 'A':      // En caso de que avanzen
                    if (collision[x + dX, y + dY] == EMPTY) {
                        collision[x, y] = EMPTY;
                        x += dX;
                        y += dY;
                        collision[x, y] = player.num;
                    }
                    break;
                case 'R':      // En caso de que retrocedan
                    if (collision[x - dX, y - dY] == EMPTY) {
                        collision[x, y] = EMPTY;
                        x -= dX;
                        y -= dY;
                        collision[x, y] = player.num;
                    }
                    break;
                case 'I':      // En caso de que giren a la izquierda
                    player.direction = (NUM_DIR + player.direction - 1) % NUM_DIR;
                    break;
                case 'D':      // En caso de que giren a la derecha
                    player.direction = (player.direction + 1) % NUM_DIR;
                    break;
                case 'S':
                    for (int i = 1; i <= SHOT_RANGE; ++i) {
                        int currCell = collision[x + i*dX, y + i*dY];
                        if (currCell == WALL) {
                            break;
                        }
                        if (currCell != EMPTY) {
                            player.kills++;
                            playerKilled = currPlayers[currCell -1];
                            playerKilled.deads++;
                            respawn_tank(playerKilled);
                            text.WriteLine("{0} {1} {2}", playerKilled.num -1, playerKilled.x, playerKilled.y);
                        }
                    }
                    text.WriteLine(-1);
                    break;
            }

            player.x = x;
            player.y = y;
        }

        /*
         * Corremos los programas
         */
        void run_programs(int turn, int totalTurns, StreamWriter text, StreamWriter errorFile) {
            for (int i = 0; i < NUM_PLAYERS_SCREEN; ++i) {
                // Escibimos la situacion actual de la batalla
                FileInfo fInfo = new FileInfo(DATA_FILE);
                StreamWriter dataFile = fInfo.CreateText();
                dataFile.WriteLine(currPlayers[i].kills - currPlayers[i].deads + " " +
                    currPlayers[i].x + " " + currPlayers[i].y + " " + direction[currPlayers[i].direction]);
                for (int j = 0; j < NUM_PLAYERS_SCREEN; ++j) {
                    if (i != j) {
                        dataFile.WriteLine(currPlayers[j].kills -currPlayers[j].deads + " " +
                            currPlayers[j].x + " " + currPlayers[j].y + " " + direction[currPlayers[j].direction]);
                    }
                }
                dataFile.Close();

                // Creamos el archivo de salida (para tenerlo listo y en limpio)
                fInfo = new FileInfo(ACTION_FILE);
                dataFile = fInfo.CreateText();
                dataFile.Close();

                // Corremos el programa del jugador
                try {
                    ProcessStartInfo processInfo = new ProcessStartInfo();
                    processInfo.WorkingDirectory = PROGRAM_FOLDER;
                    processInfo.FileName = PROGRAM_FOLDER + currPlayers[i].program;
                    processInfo.CreateNoWindow = true;
                    processInfo.UseShellExecute = false;
                    Process process = Process.Start(processInfo);

                    // Esperamos a que el proceso se cierre o se agote el tiempo
                    int timeOut = (int)(1000 * (float)programTime.Value);
                    process.WaitForExit(timeOut);
                    // Vemos si termino de correr el programa
                    if (!process.HasExited) {
                        numErrors++;
                        errorFile.WriteLine("{0}: El programa no alcanz a terminar. (Turno: {1}/{2})",
                            currPlayers[i].program, turn, totalTurns);
                        // Lo cerramos
                        if (process.Responding) {
                            process.CloseMainWindow();
                        } else {
                            process.Kill();
                        }
                    }
                } catch (Exception e) {
                    numErrors++;
                    errorFile.WriteLine("{0}: {1} (Turno: {2}/{3})",
                        currPlayers[i].program, e.Message, turn, totalTurns);
                }

                // Leemos la respuesta
                string action = "";
                if (File.Exists(ACTION_FILE)) {
                    StreamReader actionFile = File.OpenText(ACTION_FILE);
                    action = actionFile.ReadLine();
                    actionFile.Close();
                }

                // Ejecutamos la accion que eligi
                Random random = new Random(DateTime.Now.Millisecond);
                if (action == null) {
                    action = instruction[random.Next(NUM_INSTRUCTIONS-2)].ToString();
                    numErrors++;
                    errorFile.WriteLine("{0}: El programa no escribi instruccin alguna. (Turno: {1}/{2})",
                        currPlayers[i].program, turn, totalTurns);
                }
                char instr = ' ';
                for (int j = 0; j < instruction.Length; ++j) {
                    if (action.Contains(instruction[j].ToString())) {
                        instr = instruction[j];
                        break;
                    }
                }
                if (instr == ' ') {
                    instr = instruction[random.Next(NUM_INSTRUCTIONS-2)];
                    numErrors++;
                    errorFile.WriteLine("{0}: El programa no escribi alguna instruccin vlida. (Salida: '{3}') (Turno: {1}/{2})",
                        currPlayers[i].program, turn, totalTurns, action);
                }
                text.WriteLine(instr);
                update_actions(currPlayers[i], instr, text);
            }
        }

        /*
         * Calculamos las batallas
         */
        private void button1_Click(object sender, EventArgs e) {
            int numPlayers = playersInfoDataGrid.RowCount;

            // Revisamos que con la cantidad de tanques se puedan hacer los clculos
            if (numPlayers < NUM_PLAYERS_SCREEN) {
                MessageBox.Show("La cantidad de jugadores debe ser por lo menos " + NUM_PLAYERS_SCREEN.ToString(),
                    "Error!", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            if (numPlayers % NUM_PLAYERS_SCREEN != 0) {
                MessageBox.Show("La cantidad de jugadores debe ser mltiplo de " + NUM_PLAYERS_SCREEN.ToString(),
                    "Error!", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }

            // Creamos a los jugadores
            int i = 0;
            players = new PlayerData[numPlayers];
            foreach (DataGridViewRow currRow in playersInfoDataGrid.Rows) {
                players[i] = new PlayerData();
                players[i].name = (string)currRow.Cells["Alias"].Value;
                players[i].program = (string)currRow.Cells["Program"].Value;
                ++i;
            }

            // Preparamos datos para las batallas
            numErrors = 0;
            DateTime startTime = DateTime.Now;
            FileInfo fInfo = new FileInfo(ERROR_FILE);
            StreamWriter errorFile = fInfo.CreateText();
            fInfo = new FileInfo(BATTLE_FILE);
            StreamWriter text = fInfo.CreateText();
            text.WriteLine(numPlayers + " " + matchCount.Value + " " + turnsCount.Value + " " + turnDelay.Value);
            int totalMatches = (int)matchCount.Value*(numPlayers / NUM_PLAYERS_SCREEN);
            matchProgressBar.Minimum = 0;
            matchProgressBar.Maximum = (int)turnsCount.Value;
            totalProgressBar.Minimum = 0;
            totalProgressBar.Maximum = totalMatches;

            // Corremos las batallas
            randomize_matches(totalMatches);
            for (i = 0; i < totalMatches; ++i) {
                totalProgressBar.Value = i;
                initialize_match(i, text, errorFile);
                for (int j = 0; j < turnsCount.Value; ++j) {
                    matchProgressBar.Value = j;
                    run_programs(j+1, (int)turnsCount.Value, text, errorFile);
                }
                matchProgressBar.Value = (int)turnsCount.Value;
            }
            totalProgressBar.Value = totalMatches;
            text.Close();
            errorFile.Close();

            // Avisamos que ya terminamos
            TimeSpan duration =  DateTime.Now - startTime;
            MessageBox.Show("Archivo de encuentros generado.\n" + 
                "Tiempo: " + duration.Hours + "h " +  duration.Minutes + "m " + duration.Seconds + "s\n" + 
                "Errores: " + numErrors,
                "Terminado", MessageBoxButtons.OK, MessageBoxIcon.Information);

            // Reseteamos valores
            matchProgressBar.Value = 0;
            totalProgressBar.Value = 0;
        }

        /*
         * Carga todos los programas que estan en la carpeta de programas
         */
        private void cargarProgramasToolStripMenuItem_Click(object sender, EventArgs e) {
            // Vemos si existe la carpeta
            if (!Directory.Exists(PROGRAM_FOLDER)) {
                MessageBox.Show("No se encontr la carpeta de programas.\n", "Error!",
                     MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            // Buscamos los programas y los enlistamos
            string[] files = Directory.GetFiles(PROGRAM_FOLDER, "*.exe");
            playersInfoDataGrid.Rows.Clear();
            for (int i = 0; i < files.Length; ++i) {
                files[i]= files[i].Remove(0, PROGRAM_FOLDER.Length);
                playersInfoDataGrid.Rows.Add(i+1, files[i],
                    files[i].Remove(files[i].Length -4, 4));
            }
        }

        /*
         * Carga los programas y los pseudonimos desde un archivo de texto
         */
        private void cargarListaToolStripMenuItem_Click(object sender, EventArgs e) {
            // Vemos si existe el archivo
            if (!File.Exists(PLAYERS_FILE)) {
                MessageBox.Show("No se pudo abrir el archivo (el archivo no existe).\n", "Error!",
                     MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            // Leemos los datos y los pasamos a la cuadricula
            StreamReader text = File.OpenText(PLAYERS_FILE);
            int count = 0;
            string input = null;
            playersInfoDataGrid.Rows.Clear();
            while ((input = text.ReadLine()) != null) {
                ++count;
                int space = input.IndexOf(" ");
                playersInfoDataGrid.Rows.Add(count, input.Substring(0, space),
                    input.Substring(space + 1, input.Length -(space+1)));
            }
            text.Close();
        }

        /*
         * Abrimos en Notepad (o en lo que tengan por default) el txt para editarlo 
         */
        private void editarListaToolStripMenuItem_Click(object sender, EventArgs e) {
            // Corremos el programa del jugador
            try {
                ProcessStartInfo processInfo = new ProcessStartInfo();
                processInfo.FileName = PLAYERS_FILE;
                Process process = Process.Start(processInfo);
            } catch (Exception exc) {
                MessageBox.Show("No se pudo abrir el archivo: " + exc.Message + "\n", "Error!",
                     MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /* 
         * Guarda los programas y los pseudonimos en un archivo de texto
         */
        private void guardarListaToolStripMenuItem_Click(object sender, EventArgs e) {
            // Vemos si existe la carpeta
            if (!Directory.Exists(PROGRAM_FOLDER)) {
                MessageBox.Show("No se encontr la carpeta de programas.\n", "Error!",
                     MessageBoxButtons.OK, MessageBoxIcon.Error);
                return;
            }
            // Vemos si no estamos sobreescribiendo
            if (File.Exists(PLAYERS_FILE)) {
                DialogResult result = MessageBox.Show("El archivo va a ser sobreescrito.\n", "Advertencia:",
                     MessageBoxButtons.OKCancel, MessageBoxIcon.Warning);
                if (result == System.Windows.Forms.DialogResult.Cancel) {
                    return;
                }
            }
            // Guardamos los datos
            FileInfo fInfo = new FileInfo(PLAYERS_FILE);
            StreamWriter text = fInfo.CreateText();
            foreach (DataGridViewRow currRow in playersInfoDataGrid.Rows) {
                text.WriteLine(currRow.Cells["Program"].Value + " " + currRow.Cells["Alias"].Value);
            }
            text.Close();
        }

        /*
         * Se sale del programa
         */
        private void salirToolStripMenuItem_Click(object sender, EventArgs e) {
            Close();
        }

        /*
         * Informacin del programa
         */
        private void acercaDeToolStripMenuItem_Click(object sender, EventArgs e) {
            MessageBox.Show("Pier Paolo Guilln Hernndez\n", "Programado por:",
                     MessageBoxButtons.OK, MessageBoxIcon.Information);
        }        
    }
}
